<?php

defined('ABSPATH') or die();
use Firebase\JWT\JWT;
use Firebase\JWT\Key;

function enable_mail_auth(WP_REST_Request $request)
{
    $token = drk_get_header_bearer_token($request);
    $user_id = drk_get_user_id_from_token($token);
    $user = get_user_by('id', $user_id);

    if (!$user) {
        return new WP_Error('invalid_otp', 'Not authorized', array('status' => 401));
    }

    $otp_email_code = drk_generate_otp_code();
    drk_enable_2fa_email($user, $otp_email_code);

    return 'ok';
}

function disable_mail_auth(WP_REST_Request $request)
{
    $token = drk_get_header_bearer_token($request);
    if(!$token)
        return new WP_Error('invalid_token', 'Invalid token', array('status' => 401));
    $user_id = drk_get_user_id_from_token($token);
    $user = get_user_by('id', $user_id);

    if (!$user) {
        return new WP_Error('invalid_otp', 'Not authorized', array('status' => 401));
    }
    delete_user_meta($user_id, 'user_otp_mail_code','');
    delete_user_meta($user_id, 'drk_email_otp_enabled','');
    return 'ok';
}

function verify_mail_auth(WP_REST_Request $request)
{
    $secret_key = defined('JWT_AUTH_SECRET_KEY') ? JWT_AUTH_SECRET_KEY : false;

    $token = $request->get_param('token');
    if(!$token)
        return new WP_Error('invalid_token', 'No token', array('status' => 401));
    $user_id = drk_get_user_id_from_token($token);
    $user = get_user_by('ID', $user_id);
    if (!$user) {
        return new WP_Error('invalid_otp', 'Not authorized', array('status' => 401));
    }

    $otpSecret = get_user_meta($user_id ,'user_otp_mail_code', true);

    if (!$otpSecret) {
        return new WP_Error('otp_is_disabled', 'Otp mail disable', array('status' => 500));
    }

    if ($otpSecret !== $request->get_param('code')) {
        return new WP_Error('invalid_otp', 'Invalid otp code', array('status' => 400));
    }else{
        $otpSecret = drk_generate_otp_code();
        update_user_meta($user->ID, 'user_otp_mail_code', $otpSecret);
    }

    $issuedAt = time();
    $notBefore = $issuedAt;
    $expire = $issuedAt + (DAY_IN_SECONDS * 7);

    $token = [
        'iss' => get_bloginfo('url'),
        'iat' => $issuedAt,
        'nbf' => $notBefore,
        'exp' => $expire,
        'data' => [
            'user' => [
                'id' => $user->ID,
            ],
        ],
    ];

    $email_is_confirmed = get_the_author_meta('drk_email_is_confirmed',$user->data->ID);
    $profile = drk_get_user_profile((array)$user->data);
    $roles = $user->get_role_caps();

    return [
        'token' => Firebase\JWT\JWT::encode($token, $secret_key,"HS256"),
        'user_email' => $user->data->user_email,
        'user_nicename' => $user->data->user_nicename,
        'user_display_name' => $user->data->display_name,
        "user_id" => $user->data->ID,
        "user_role" => get_userdata($user->data->ID)->roles,
        "user_caps" => $roles,
        "user_registered" => $user->data->user_registered,
        "2fa" => "Email",
        "is_email_confirmed" => $email_is_confirmed,
        "profile" => $profile
    ];
}

add_action('rest_api_init', 'get_2fa_mail_rest');
function get_2fa_mail_rest()
{

    register_rest_route('2fa/v1', '/enable-mail', [
        'methods' => 'POST',
        'callback' => 'enable_mail_auth',
        'permission_callback' => '__return_true'
    ]);

    register_rest_route('2fa/v1', '/disable-mail', [
        'methods' => 'POST',
        'callback' => 'disable_mail_auth',
        'permission_callback' => '__return_true'
    ]);

    register_rest_route('2fa/v1', '/verify-mail', [
        'methods' => 'POST',
        'callback' => 'verify_mail_auth',
        'permission_callback' => '__return_true'
    ]);

}

add_filter('jwt_auth_token_before_dispatch', 'check_mail_auth_is_enable', 10, 2);
function check_mail_auth_is_enable($data, $user)
{
    $email_otp_is_enabled = get_user_meta($user->ID, 'drk_email_otp_enabled', true);

    if ($email_otp_is_enabled !="true") {
        return $data;
    }
    $otpSecret = drk_generate_otp_code();
    update_user_meta($user->ID, 'user_otp_mail_code', $otpSecret);

    $secret_key = defined('JWT_AUTH_SECRET_KEY') ? JWT_AUTH_SECRET_KEY : false;
    $issuedAt = time();
    $notBefore = $issuedAt;
    $expire = $issuedAt + 3600;

    //drk_send_mail($user, 'Darkey: OTP code', $otpSecret);

    wp_mail($user->data->user_email,'Darkey: OTP code', $otpSecret);
    $token = [
        'iss' => get_bloginfo('url'),
        'iat' => $issuedAt,
        'nbf' => $notBefore,
        'exp' => $expire,
        'data' => [
            'user' => [
                'id' => $user->ID,
            ],
        ],
    ];

    $jwt = JWT::encode($token, $secret_key,'HS256');

    return new WP_Error('invalid_otp',
        'OTP Auth is enabled',
        ['checker' => 'mailer',
            'token' => $jwt,
            'status' => 403,
            'email_otp'=>$email_otp_is_enabled
        ]);
}
