local max_results = 30

function personal_setting_value(player, name)
  if player and player.mod_settings and player.mod_settings[name] then
    return player.mod_settings[name].value
  else
    return nil
  end
end

function get_entity_name_list_setting(player, setting_name)
  local s = personal_setting_value(player, setting_name)
  if s then
    local names = {}
    for name in string.gmatch(s, "[^,]+") do
      name = string.gsub(name, "^%s+", "")
      name = string.gsub(name, "%s+$", "")
      if string.len(name) > 0 then
        table.insert(names, name)
      end
    end
    return names
  else
    return {}
  end
end


function round(v)
  return v and math.floor(v + 0.5)
end

local compass_directions = {"east", "southeast", "south", "southwest", "west", "northwest", "north", "northeast"}
function compass_direction(dx, dy)
  if dx == 0 and dy == 0 then return nil end
  local angle = math.deg(math.atan2(dy, dx))
  local index = math.floor(((angle + 22.5) % 360) / 45)
  return compass_directions[index + 1] or nil
end

function heading(pos_from, pos_to)
  if pos_from and pos_to then
    local dx = pos_to.x - pos_from.x
    local dy = pos_to.y - pos_from.y
    return {
      distance = math.sqrt(dx*dx + dy*dy),
      direction = compass_direction(dx, dy)
    }
  else
    return {
      distance = nil,
      direction = nil
    }
  end
end

function find_vehicle_info(player, entity)
  return {
    entity = entity,
    heading = heading(player.position, entity.position),
    -- suitable for passing as a parameter when printing a localised string
    -- see https://lua-api.factorio.com/latest/Concepts.html#LocalisedString
    localised_name_token = (entity.prototype and entity.prototype.localised_name) or entity.name
  }
end

function find_vehicle_infos(player)
  local filters = {
    type = {"car", "spider-vehicle"},
    force = player.force.name,
    limit = max_results -- just to keep it from totally barfing
  }
  -- build a lookup table to make it easy to discard hidden entities
  local hidden_names = {}
  for _, name in pairs(get_entity_name_list_setting(player, "car-finder-hidden-entity-names")) do
    hidden_names[name] = true
  end
  -- add positive filter for entity name if present
  local selected_names = get_entity_name_list_setting(player, "car-finder-selected-entity-names")
  if table_size(selected_names) > 0 then
    filters.name = selected_names
  end
  -- find and build the list
  local vehicles = player.surface.find_entities_filtered(filters)
  local infos = {}
  for _, entity in pairs(vehicles) do
    if not hidden_names[entity.name] then
      table.insert(infos, find_vehicle_info(player, entity))
    end
  end
  -- sort by increasing distance
  local function sort_pred(a, b)
    return (a.heading.distance or 0) < (b.heading.distance or 0)
  end
  table.sort(infos, sort_pred)
  return infos
end

function print_vehicle_summary(player, vehicle_info)
  local entity = vehicle_info.entity
  local show_colors = personal_setting_value(player, "car-finder-setting-show-colors")
  local show_heading = personal_setting_value(player, "car-finder-setting-show-distance")
  local show_coordinates = personal_setting_value(player, "car-finder-setting-show-coordinates")

  -- the empty-string locale key is special and it means everything will get concatenated when printed
  -- https://wiki.factorio.com/Tutorial:Localisation#Concatenating_localised_strings
  local line_parts = {"", "[img=entity/" .. entity.name .. "] "}
  if show_colors and entity.color and entity.color.a > 0 then
    local r = round(255 * entity.color.r)
    local g = round(255 * entity.color.g)
    local b = round(255 * entity.color.b)
    table.insert(line_parts, "[color=" .. r .. "," .. g .. "," .. b .. "]")
    table.insert(line_parts, vehicle_info.localised_name_token)
    table.insert(line_parts, "[/color]")
  else
    table.insert(line_parts, vehicle_info.localised_name_token)
  end
  if personal_setting_value(player, "car-finder-setting-show-entity-names") then
    table.insert(line_parts, " (" .. entity.name .. ")")
  end
  -- TODO: maybe in the future, devs will give us a way to get Spidertron labels
  if show_heading or show_coordinates then
    table.insert(line_parts, " is ")
    if show_heading then
      local heading = vehicle_info.heading
      local part = (round(heading.distance) or "?") .. "m " .. (heading.direction or "away") .. " "
      table.insert(line_parts, part)
    end
    if show_coordinates then
      local part = " at [gps=" .. round(entity.position.x) .. "," .. round(entity.position.y) .. "]"
      table.insert(line_parts, part)
    end
  end
  player.print(line_parts)
end

function maybe_play_confirmation_sound(player)
  local volume = personal_setting_value(player, "car-finder-setting-chirp-volume")
  if volume and volume > 0 then
    player.play_sound({ path = "car-finder-activated", volume_modifier = volume })
  end
end

function print_all_vehicles(player)
  local vehicle_infos = find_vehicle_infos(player)
  player.print({"car-finder.notify-searching"})
  local count = 0
  for _, vehicle_info in pairs(vehicle_infos) do
    print_vehicle_summary(player, vehicle_info)
    count = count + 1
  end
  if count == 0 then
    player.print({"car-finder.result-none"})
  elseif count >= max_results then
    player.print({"car-finder.result-overflow"})
  end
end

function maybe_focus_linked_spidertron(player)
  local focus_enabled = personal_setting_value(player, "car-finder-focus-linked-remote")
  local remote = player.cursor_stack
  -- TODO: relax the restriction on type and allow for any kind of linked entity? IDK what mods use this
  if focus_enabled and remote and remote.valid and remote.valid_for_read and remote.type == "spidertron-remote" then
    local vehicle = remote.connected_entity
    if vehicle and vehicle.valid and vehicle.surface == player.surface then
      player.print({"car-finder.result-spider-focus"})
      print_vehicle_summary(player, find_vehicle_info(player, vehicle))
      -- omit scale and game will keep player zoomed how they are zoomed
      if player.render_mode == defines.render_mode.chart then
        player.open_map(vehicle.position)
      else
        player.zoom_to_world(vehicle.position)
      end
      return true
    end
  end
  return false
end

function activate_car_finder(player)
  if player and player.valid and player.surface then
    local focused = maybe_focus_linked_spidertron(player)
    if not focused then
      print_all_vehicles(player, vehicle_infos)
    end
    maybe_play_confirmation_sound(player)
  end
end


-- user clicked button directly
script.on_event(defines.events.on_lua_shortcut, function(event)
  if event and event.prototype_name == "car-finder-button" then
    activate_car_finder(game.players[event.player_index])
  end
end)

-- user triggered keyboard shortcut
script.on_event("car-finder-hotkey", function(event)
  if event then
    activate_car_finder(game.players[event.player_index])
  end
end)
